# project dependencies
from lightecc.interfaces.form import WeierstrassInterface
from lightecc.commons.logger import Logger

logger = Logger(module="lightecc/curves/weierstrass.py")


DEFAULT_CURVE = "secp256k1"


# pylint: disable=too-few-public-methods
class Secp256k1(WeierstrassInterface):
    a = 0
    b = 7
    p = (
        pow(2, 256)
        - pow(2, 32)
        - pow(2, 9)
        - pow(2, 8)
        - pow(2, 7)
        - pow(2, 6)
        - pow(2, 4)
        - pow(2, 0)
    )
    G = (
        55066263022277343669578718895168534326250603453777594175500187360389116729240,
        32670510020758816978083085130507043184471273380659243275938904335757337482424,
    )
    n = 115792089237316195423570985008687907852837564279074904382605163141518161494337


class P192(WeierstrassInterface):
    a = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFC
    b = 0x64210519E59C80E70FA7E9AB72243049FEB8DEECC146B9B1
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFF
    G = (
        0x188DA80EB03090F67CBF20EB43A18800F4FF0AFD82FF1012,
        0x07192B95FFC8DA78631011ED6B24CDD573F977A11E794811,
    )
    n = 0xFFFFFFFFFFFFFFFFFFFFFFFF99DEF836146BC9B1B4D22831


class P224(WeierstrassInterface):
    a = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFE
    b = 0xB4050A850C04B3ABF54132565044B0B7D7BFD8BA270B39432355FFB4
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000000000000000000000001
    G = (
        0xB70E0CBD6BB4BF7F321390B94A03C1D356C21122343280D6115C1D21,
        0xBD376388B5F723FB4C22DFE6CD4375A05A07476444D5819985007E34,
    )
    n = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFF16A2E0B8F03E13DD29455C5C2A3D


class P256(WeierstrassInterface):
    a = 0xFFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFC
    b = 0x5AC635D8AA3A93E7B3EBBD55769886BC651D06B0CC53B0F63BCE3C3E27D2604B
    p = 0xFFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFF
    G = (
        0x6B17D1F2E12C4247F8BCE6E563A440F277037D812DEB33A0F4A13945D898C296,
        0x4FE342E2FE1A7F9B8EE7EB4A7C0F9E162BCE33576B315ECECBB6406837BF51F5,
    )
    n = 0xFFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632551


class P384(WeierstrassInterface):
    a = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF0000000000000000FFFFFFFC
    b = 0xB3312FA7E23EE7E4988E056BE3F82D19181D9C6EFE8141120314088F5013875AC656398D8A2ED19D2A85C8EDD3EC2AEF
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF0000000000000000FFFFFFFF
    G = (
        0xAA87CA22BE8B05378EB1C71EF320AD746E1D3B628BA79B9859F741E082542A385502F25DBF55296C3A545E3872760AB7,
        0x3617DE4A96262C6F5D9E98BF9292DC29F8F41DBD289A147CE9DA3113B5F0B8C00A60B1CE1D7E819D7A431D7C90EA0E5F,
    )
    n = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC7634D81F4372DDF581A0DB248B0A77AECEC196ACCC52973


class P521(WeierstrassInterface):
    a = 0x01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC
    b = 0x0051953EB9618E1C9A1F929A21A0B68540EEA2DA725B99B315F3B8B489918EF109E156193951EC7E937B1652C0BD3BB1BF073573DF883D2C34F1EF451FD46B503F00
    p = 0x01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
    G = (
        0x00C6858E06B70404E9CD9E3ECB662395B4429C648139053FB521F828AF606B4D3DBAA14B5E77EFE75928FE1DC127A2FFA8DE3348B3C1856A429BF97E7E31C2E5BD66,
        0x011839296A789A3BC0045C8A5FB42C7D1BD998F54449579B446817AFBD17273E662C97EE72995EF42640C550B9013FAD0761353C7086A272C24088BE94769FD16650,
    )
    n = 0x01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFA51868783BF2F966B7FCC0148F709A5D03BB5C9B8899C47AEBB6FB71E91386409


class Curve22103(WeierstrassInterface):
    a = 0x155555555555555555555555555555555555555555555552174084FF
    b = 0x1425ED097B425ED097B425ED097B425ED097B425ED0BBA9428427967
    p = 0x1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFD
    G = (
        0xCB476FE081B61F56A60B1D1B34FB7207D072EF25DADF377731049B0,
        0x36429404D97E1E217BAD2E5601F6551F95D8FE9481BD454D1F3E7B6,
    )
    n = 0x3FFFFFFFFFFFFFFFFFFFFFFFFFFF5CD04695A145C3067CF4AAE2025


class Curve4417(WeierstrassInterface):
    a = 0x4648D10B419379D50F4BA01869D9AE363285E01FE66920878EE075B0
    b = 0x4C4DCEDFAC09383A0311B98EE9637415B9134B4115FDB760C1A3D419
    p = 0x3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFB
    G = (
        0xFF9D7EDD97D523012FFAACA32B7C90F1A8E026F21972C789A8484A2B,
        0x3E9036ADC1A41FCE2F2CA08E2D1BE4F6D97E30CA7761DB1F3E2F2CE96,
    )
    n = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC4A75594D4923FC93D42713CDAF


class Curve1174(WeierstrassInterface):
    a = 0x486BE25B34C8080922B969257EEB54C404F914A29067A5560BB9AEE0BC67A6D
    b = 0xE347A25BF875DD2F1F12D8A10334D417CC15E77893A99F4BF278CA563072E6
    p = 0x7FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7
    G = (
        0x3BE821D63D2CD5AFE0504F452E5CF47A60A10446928CEAECFD5294F89B45051,
        0x66FE4E7B8B6FE152F743393029A61BFB839747C8FB00F7B27A6841C07532A0,
    )
    n = 0x1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF77965C4DFD307348944D45FD166C971


class Curve67254(WeierstrassInterface):
    a = 0x2E32419A32377AF8E7F03148A106D112C8C2E26D31A89F46B743DEED322C7ADC3024AFE4B5AFD8DB7180281586549F4A
    b = 0x22F6EF3BE72A67FDC236D7173727CD2AF6D02A195753C44BDF451369B02EA0F963D9A775CAE6DC3AE9CCABB7F183C1AD
    p = 0x3FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF97
    G = (
        0x141946416199FDDA96EC1F1AC80058AA3F0CE96B1BCD023E97BAE542FE3404B4738DB9B81A13698339387672ECA360C0,
        0xD51BF79D968F4A076022E750F821058E2B5073697B639EDD355EBF8AD32352B1EFA9478DE7EB5662EF0D26EF6EEA795,
    )
    n = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFD5FB21F21E95EEE17C5E69281B102D2773E27E13FD3C9719


class Fp254BNa(WeierstrassInterface):
    a = 0
    b = 0x05
    p = 0x2370FB049D410FBE4E761A9886E502417D023F40180000017E80600000000001
    G = (0x01, 0xD45589B158FAAF6AB0E4AD38D998E9982E7FF63964EE1460342A592677CCCB0)
    n = 0x2370FB049D410FBE4E761A9886E502411DC1AF70120000017E80600000000001


class Fp254BNb(WeierstrassInterface):
    a = 0x00
    b = 0x02
    p = 0x2523648240000001BA344D80000000086121000000000013A700000000000013
    G = (0x2523648240000001BA344D80000000086121000000000013A700000000000012, 0x01)
    n = 0x2523648240000001BA344D8000000007FF9F800000000010A10000000000000D


class Fp224BN(WeierstrassInterface):
    a = 0
    b = 0x03
    p = 0xFFFFFFFFFFF107288EC29E602C4520DB42180823BB907D1287127833
    G = (0x01, 0x02)
    n = 0xFFFFFFFFFFF107288EC29E602C4420DB4218082B36C2ACCFF76C58ED


class Fp256BN(WeierstrassInterface):
    a = 0
    b = 0x03
    p = 0xFFFFFFFFFFFCF0CD46E5F25EEE71A49F0CDC65FB12980A82D3292DDBAED33013
    G = (0x01, 0x02)
    n = 0xFFFFFFFFFFFCF0CD46E5F25EEE71A49E0CDC65FB1299921AF62D536CD10B500D


class Fp384BN(WeierstrassInterface):
    a = 0
    b = 0x03
    p = 0xFFFFFFFFFFFFFFFFFFF2A96823D5920D2A127E3F6FBCA024C8FBE29531892C79534F9D306328261550A7CABD7CCCD10B
    G = (0x01, 0x02)
    n = 0xFFFFFFFFFFFFFFFFFFF2A96823D5920D2A127E3F6FBCA023C8FBE29531892C795356487D8AC63E4F4DB17384341A5775


class Fp512BN(WeierstrassInterface):
    a = 0
    b = 0x03
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFF9EC7F01C60BA1D8CB5307C0BBE3C111B0EF455146CF1EACBE98B8E48C65DEAB236FE1916A55CE5F4C6467B4EB280922ADEF33
    G = (0x01, 0x02)
    n = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFF9EC7F01C60BA1D8CB5307C0BBE3C111B0EF445146CF1EACBE98B8E48C65DEAB2679A34A10313E04F9A2B406A64A5F519A09ED


class Tweedledum(WeierstrassInterface):
    a = 0
    b = 5
    p = 0x40000000000000000000000000000000038AA1276C3F59B9A14064E200000001
    G = (0x40000000000000000000000000000000038AA1276C3F59B9A14064E200000000, 0x02)
    n = 0x40000000000000000000000000000000038AA127696286C9842CAFD400000001


class Tweedledee(WeierstrassInterface):
    a = 0
    b = 5
    p = 0x40000000000000000000000000000000038AA127696286C9842CAFD400000001
    G = (0x40000000000000000000000000000000038AA127696286C9842CAFD400000000, 0x02)
    n = 0x40000000000000000000000000000000038AA1276C3F59B9A14064E200000001


class Pallas(WeierstrassInterface):
    a = 0x00
    b = 0x05
    p = 0x40000000000000000000000000000000224698FC094CF91B992D30ED00000001
    G = (0x40000000000000000000000000000000224698FC094CF91B992D30ED00000000, 0x02)
    n = 0x40000000000000000000000000000000224698FC0994A8DD8C46EB2100000001


class Vesta(WeierstrassInterface):
    a = 0x00
    b = 0x05
    p = 0x40000000000000000000000000000000224698FC0994A8DD8C46EB2100000001
    G = (0x40000000000000000000000000000000224698FC0994A8DD8C46EB2100000000, 0x02)
    n = 0x40000000000000000000000000000000224698FC094CF91B992D30ED00000001


class Tom256(WeierstrassInterface):
    a = 0xFFFFFFFF0000000100000000000000017E72B42B30E7317793135661B1C4B114
    b = 0xB441071B12F4A0366FB552F8E21ED4AC36B06ACEEB354224863E60F20219FC56
    p = 0xFFFFFFFF0000000100000000000000017E72B42B30E7317793135661B1C4B117
    G = (0x03, 0x5A6DD32DF58708E64E97345CBE66600DECD9D538A351BB3C30B4954925B1F02D)
    n = 0xFFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFF


class Numsp256d1(WeierstrassInterface):
    a = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF40
    b = 0x25581
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF43
    G = (0x01, 0x696F1853C1E466D7FC82C96CCEEEDD6BD02C2F9375894EC10BF46306C2B56C77)
    n = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFE43C8275EA265C6020AB20294751A825


class Numsp384d1(WeierstrassInterface):
    a = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEC0
    b = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF77BB
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEC3
    G = (
        0x02,
        0x3C9F82CB4B87B4DC71E763E0663E5DBD8034ED422F04F82673330DC58D15FFA2B4A3D0BAD5D30F865BCBBF503EA66F43,
    )
    n = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFD61EAF1EEB5D6881BEDA9D3D4C37E27A604D81F67B0E61B9


class Numsp512d1(WeierstrassInterface):
    a = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFDC4
    b = 0x1D99B
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFDC7
    G = (
        0x02,
        0x1C282EB23327F9711952C250EA61AD53FCC13031CF6DD336E0B9328433AFBDD8CC5A1C1F0C716FDC724DDE537C2B0ADB00BB3D08DC83755B205CC30D7F83CF28,
    )
    n = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF5B3CA4FB94E7831B4FC258ED97D0BDC63B568B36607CD243CE153F390433555D


class BrainpoolP160r1(WeierstrassInterface):
    a = 0x340E7BE2A280EB74E2BE61BADA745D97E8F7C300
    b = 0x1E589A8595423412134FAA2DBDEC95C8D8675E58
    p = 0xE95E4A5F737059DC60DFC7AD95B3D8139515620F
    G = (
        0xBED5AF16EA3F6A4F62938C4631EB5AF7BDBCDBC3,
        0x1667CB477A1A8EC338F94741669C976316DA6321,
    )
    n = 0xE95E4A5F737059DC60DF5991D45029409E60FC09


class BrainpoolP160t1(WeierstrassInterface):
    a = 0xE95E4A5F737059DC60DFC7AD95B3D8139515620C
    b = 0x7A556B6DAE535B7B51ED2C4D7DAA7A0B5C55F380
    p = 0xE95E4A5F737059DC60DFC7AD95B3D8139515620F
    G = (
        0xB199B13B9B34EFC1397E64BAEB05ACC265FF2378,
        0xADD6718B7C7C1961F0991B842443772152C9E0AD,
    )
    n = 0xE95E4A5F737059DC60DF5991D45029409E60FC09


class BrainpoolP192r1(WeierstrassInterface):
    a = 0x6A91174076B1E0E19C39C031FE8685C1CAE040E5C69A28EF
    b = 0x469A28EF7C28CCA3DC721D044F4496BCCA7EF4146FBF25C9
    p = 0xC302F41D932A36CDA7A3463093D18DB78FCE476DE1A86297
    G = (
        0xC0A0647EAAB6A48753B033C56CB0F0900A2F5C4853375FD6,
        0x14B690866ABD5BB88B5F4828C1490002E6773FA2FA299B8F,
    )
    n = 0xC302F41D932A36CDA7A3462F9E9E916B5BE8F1029AC4ACC1


class BrainpoolP192t1(WeierstrassInterface):
    a = 0xC302F41D932A36CDA7A3463093D18DB78FCE476DE1A86294
    b = 0x13D56FFAEC78681E68F9DEB43B35BEC2FB68542E27897B79
    p = 0xC302F41D932A36CDA7A3463093D18DB78FCE476DE1A86297
    G = (
        0x3AE9E58C82F63C30282E1FE7BBF43FA72C446AF6F4618129,
        0x97E2C5667C2223A902AB5CA449D0084B7E5B3DE7CCC01C9,
    )
    n = 0xC302F41D932A36CDA7A3462F9E9E916B5BE8F1029AC4ACC1


class BrainpoolP224r1(WeierstrassInterface):
    a = 0x68A5E62CA9CE6C1C299803A6C1530B514E182AD8B0042A59CAD29F43
    b = 0x2580F63CCFE44138870713B1A92369E33E2135D266DBB372386C400B
    p = 0xD7C134AA264366862A18302575D1D787B09F075797DA89F57EC8C0FF
    G = (
        0xD9029AD2C7E5CF4340823B2A87DC68C9E4CE3174C1E6EFDEE12C07D,
        0x58AA56F772C0726F24C6B89E4ECDAC24354B9E99CAA3F6D3761402CD,
    )
    n = 0xD7C134AA264366862A18302575D0FB98D116BC4B6DDEBCA3A5A7939F


class BrainpoolP224t1(WeierstrassInterface):
    a = 0xD7C134AA264366862A18302575D1D787B09F075797DA89F57EC8C0FC
    b = 0x4B337D934104CD7BEF271BF60CED1ED20DA14C08B3BB64F18A60888D
    p = 0xD7C134AA264366862A18302575D1D787B09F075797DA89F57EC8C0FF
    G = (
        0x6AB1E344CE25FF3896424E7FFE14762ECB49F8928AC0C76029B4D580,
        0x374E9F5143E568CD23F3F4D7C0D4B1E41C8CC0D1C6ABD5F1A46DB4C,
    )
    n = 0xD7C134AA264366862A18302575D0FB98D116BC4B6DDEBCA3A5A7939F


class BrainpoolP256r1(WeierstrassInterface):
    a = 0x7D5A0975FC2C3057EEF67530417AFFE7FB8055C126DC5C6CE94A4B44F330B5D9
    b = 0x26DC5C6CE94A4B44F330B5D9BBD77CBF958416295CF7E1CE6BCCDC18FF8C07B6
    p = 0xA9FB57DBA1EEA9BC3E660A909D838D726E3BF623D52620282013481D1F6E5377
    G = (
        0x8BD2AEB9CB7E57CB2C4B482FFC81B7AFB9DE27E1E3BD23C23A4453BD9ACE3262,
        0x547EF835C3DAC4FD97F8461A14611DC9C27745132DED8E545C1D54C72F046997,
    )
    n = 0xA9FB57DBA1EEA9BC3E660A909D838D718C397AA3B561A6F7901E0E82974856A7


class BrainpoolP256t1(WeierstrassInterface):
    a = 0xA9FB57DBA1EEA9BC3E660A909D838D726E3BF623D52620282013481D1F6E5374
    b = 0x662C61C430D84EA4FE66A7733D0B76B7BF93EBC4AF2F49256AE58101FEE92B04
    p = 0xA9FB57DBA1EEA9BC3E660A909D838D726E3BF623D52620282013481D1F6E5377
    G = (
        0xA3E8EB3CC1CFE7B7732213B23A656149AFA142C47AAFBC2B79A191562E1305F4,
        0x2D996C823439C56D7F7B22E14644417E69BCB6DE39D027001DABE8F35B25C9BE,
    )
    n = 0xA9FB57DBA1EEA9BC3E660A909D838D718C397AA3B561A6F7901E0E82974856A7


class BrainpoolP320r1(WeierstrassInterface):
    a = 0x3EE30B568FBAB0F883CCEBD46D3F3BB8A2A73513F5EB79DA66190EB085FFA9F492F375A97D860EB4
    b = 0x520883949DFDBC42D3AD198640688A6FE13F41349554B49ACC31DCCD884539816F5EB4AC8FB1F1A6
    p = 0xD35E472036BC4FB7E13C785ED201E065F98FCFA6F6F40DEF4F92B9EC7893EC28FCD412B1F1B32E27
    G = (
        0x43BD7E9AFB53D8B85289BCC48EE5BFE6F20137D10A087EB6E7871E2A10A599C710AF8D0D39E20611,
        0x14FDD05545EC1CC8AB4093247F77275E0743FFED117182EAA9C77877AAAC6AC7D35245D1692E8EE1,
    )
    n = 0xD35E472036BC4FB7E13C785ED201E065F98FCFA5B68F12A32D482EC7EE8658E98691555B44C59311


class BrainpoolP320t1(WeierstrassInterface):
    a = 0xD35E472036BC4FB7E13C785ED201E065F98FCFA6F6F40DEF4F92B9EC7893EC28FCD412B1F1B32E24
    b = 0xA7F561E038EB1ED560B3D147DB782013064C19F27ED27C6780AAF77FB8A547CEB5B4FEF422340353
    p = 0xD35E472036BC4FB7E13C785ED201E065F98FCFA6F6F40DEF4F92B9EC7893EC28FCD412B1F1B32E27
    G = (
        0x925BE9FB01AFC6FB4D3E7D4990010F813408AB106C4F09CB7EE07868CC136FFF3357F624A21BED52,
        0x63BA3A7A27483EBF6671DBEF7ABB30EBEE084E58A0B077AD42A5A0989D1EE71B1B9BC0455FB0D2C3,
    )
    n = 0xD35E472036BC4FB7E13C785ED201E065F98FCFA5B68F12A32D482EC7EE8658E98691555B44C59311


class BrainpoolP384r1(WeierstrassInterface):
    a = 0x7BC382C63D8C150C3C72080ACE05AFA0C2BEA28E4FB22787139165EFBA91F90F8AA5814A503AD4EB04A8C7DD22CE2826
    b = 0x4A8C7DD22CE28268B39B55416F0447C2FB77DE107DCD2A62E880EA53EEB62D57CB4390295DBC9943AB78696FA504C11
    p = 0x8CB91E82A3386D280F5D6F7E50E641DF152F7109ED5456B412B1DA197FB71123ACD3A729901D1A71874700133107EC53
    G = (
        0x1D1C64F068CF45FFA2A63A81B7C13F6B8847A3E77EF14FE3DB7FCAFE0CBD10E8E826E03436D646AAEF87B2E247D4AF1E,
        0x8ABE1D7520F9C2A45CB1EB8E95CFD55262B70B29FEEC5864E19C054FF99129280E4646217791811142820341263C5315,
    )
    n = 0x8CB91E82A3386D280F5D6F7E50E641DF152F7109ED5456B31F166E6CAC0425A7CF3AB6AF6B7FC3103B883202E9046565


class BrainpoolP384t1(WeierstrassInterface):
    a = 0x8CB91E82A3386D280F5D6F7E50E641DF152F7109ED5456B412B1DA197FB71123ACD3A729901D1A71874700133107EC50
    b = 0x7F519EADA7BDA81BD826DBA647910F8C4B9346ED8CCDC64E4B1ABD11756DCE1D2074AA263B88805CED70355A33B471EE
    p = 0x8CB91E82A3386D280F5D6F7E50E641DF152F7109ED5456B412B1DA197FB71123ACD3A729901D1A71874700133107EC53
    G = (
        0x18DE98B02DB9A306F2AFCD7235F72A819B80AB12EBD653172476FECD462AABFFC4FF191B946A5F54D8D0AA2F418808CC,
        0x25AB056962D30651A114AFD2755AD336747F93475B7A1FCA3B88F2B6A208CCFE469408584DC2B2912675BF5B9E582928,
    )
    n = 0x8CB91E82A3386D280F5D6F7E50E641DF152F7109ED5456B31F166E6CAC0425A7CF3AB6AF6B7FC3103B883202E9046565


class BrainpoolP512r1(WeierstrassInterface):
    a = 0x7830A3318B603B89E2327145AC234CC594CBDD8D3DF91610A83441CAEA9863BC2DED5D5AA8253AA10A2EF1C98B9AC8B57F1117A72BF2C7B9E7C1AC4D77FC94CA
    b = 0x3DF91610A83441CAEA9863BC2DED5D5AA8253AA10A2EF1C98B9AC8B57F1117A72BF2C7B9E7C1AC4D77FC94CADC083E67984050B75EBAE5DD2809BD638016F723
    p = 0xAADD9DB8DBE9C48B3FD4E6AE33C9FC07CB308DB3B3C9D20ED6639CCA703308717D4D9B009BC66842AECDA12AE6A380E62881FF2F2D82C68528AA6056583A48F3
    G = (
        0x81AEE4BDD82ED9645A21322E9C4C6A9385ED9F70B5D916C1B43B62EEF4D0098EFF3B1F78E2D0D48D50D1687B93B97D5F7C6D5047406A5E688B352209BCB9F822,
        0x7DDE385D566332ECC0EABFA9CF7822FDF209F70024A57B1AA000C55B881F8111B2DCDE494A5F485E5BCA4BD88A2763AED1CA2B2FA8F0540678CD1E0F3AD80892,
    )
    n = 0xAADD9DB8DBE9C48B3FD4E6AE33C9FC07CB308DB3B3C9D20ED6639CCA70330870553E5C414CA92619418661197FAC10471DB1D381085DDADDB58796829CA90069


class BrainpoolP512t1(WeierstrassInterface):
    a = 0xAADD9DB8DBE9C48B3FD4E6AE33C9FC07CB308DB3B3C9D20ED6639CCA703308717D4D9B009BC66842AECDA12AE6A380E62881FF2F2D82C68528AA6056583A48F0
    b = 0x7CBBBCF9441CFAB76E1890E46884EAE321F70C0BCB4981527897504BEC3E36A62BCDFA2304976540F6450085F2DAE145C22553B465763689180EA2571867423E
    p = 0xAADD9DB8DBE9C48B3FD4E6AE33C9FC07CB308DB3B3C9D20ED6639CCA703308717D4D9B009BC66842AECDA12AE6A380E62881FF2F2D82C68528AA6056583A48F3
    G = (
        0x640ECE5C12788717B9C1BA06CBC2A6FEBA85842458C56DDE9DB1758D39C0313D82BA51735CDB3EA499AA77A7D6943A64F7A3F25FE26F06B51BAA2696FA9035DA,
        0x5B534BD595F5AF0FA2C892376C84ACE1BB4E3019B71634C01131159CAE03CEE9D9932184BEEF216BD71DF2DADF86A627306ECFF96DBB8BACE198B61E00F8B332,
    )
    n = 0xAADD9DB8DBE9C48B3FD4E6AE33C9FC07CB308DB3B3C9D20ED6639CCA70330870553E5C414CA92619418661197FAC10471DB1D381085DDADDB58796829CA90069


class Mnt1(WeierstrassInterface):
    a = 0x22FFBB20CC052993FA27DC507800B624C650E4FF3D2
    b = 0x1C7BE6FA8DA953B5624EFC72406AF7FA77499803D08
    p = 0x26DCCACC5041939206CF2B7DEC50950E3C9FA4827AF
    G = (
        0x25A3AE778F7EF6586ABAE5ACDE21E54B6C64EDF33D0,
        0x05B4ACE33AA53C670CE35535D6C273698A182DA557D,
    )
    n = 0x0000A60FD646AD409B3312C3B23BA64E082AD7B354D


class Mnt2_1(WeierstrassInterface):
    a = 0x3DD24A7E5C0BDFACCC215E22760469C73EE9D879
    b = 0x478C31A992B294E19F6E4416F958646DDDEDE5E3
    p = 0x5AFFFFFFFFFFFF4B46081000000059BB1BF600B7
    G = (
        0x2725AF3D7DEA98CB9242AC6DDB9BD89BDCF38898,
        0x480B4184ED2C50C0230B4C73CA939C1B6B7F1103,
    )
    n = 0x5AFFFFFFFFFFFF4B46081000000059BB1BF600B5


class Mnt2_2(WeierstrassInterface):
    a = 0x07B29491C1A02CD87844F5098D0381F6C45D6523
    b = 0x41CC630BD66AC817D43358B108AD3D214037993C
    p = 0x5AFFFFFFFFFFFF4B46081000000059BB1BF600B7
    G = (
        0x0D76B3E1F1ED76A282FA99575D29FF2E587049E9,
        0x36E1557ED145AD409F924420E12F74A900FAB054,
    )
    n = 0x5AFFFFFFFFFFFF4B46081000000059BB1BF600B5


class Mnt3_1(WeierstrassInterface):
    a = 0x6D01FD0A017C62075AE999977379867E07F2A6D4
    b = 0x7701535C00FD965341D38BBA4CFBDCF9A4651825
    p = 0x8AFFFFFFFFFFFEEB0FA77000000089F0DD49FAC7
    G = (
        0x1781998103C3CA14EA76B9D3A700A53E1C784789,
        0x53352DDE04447C25C9BB332A3C7634D3B8801F34,
    )
    n = 0x8AFFFFFFFFFFFEEB0FA77000000089F0DD49FAC5


class Mnt3_2(WeierstrassInterface):
    a = 0x5FBE0085BD2B23AFCD5B9C7704AEED2BFDBE89E4
    b = 0x3FD4005928C76D1FDE3D12FA031F48C7FE7F0698
    p = 0x8AFFFFFFFFFFFEEB0FA77000000089F0DD49FAC7
    G = (
        0x494E297179D42C761701AB03B2E5BCA98A24DFE7,
        0x3274201D6596252A780390A222E3763BBECFE5F1,
    )
    n = 0x8AFFFFFFFFFFFEEB0FA77000000089F0DD49FAC5


class Mnt3_3(WeierstrassInterface):
    a = 0x2DDF23ACB05A91BDA6BA9C20D7A584AA25075CE0
    b = 0x1F8125C46A31E79FD6CC25298B23AB130CD22B5A
    p = 0x8AFFFFFFFFFFFEEB0FA77000000089F0DD49FAC7
    G = (
        0x3F710D05B65B5E16AE1B946D3FC582B16A927432,
        0x4A30945C64FD7F85E148BA816005468447616B1F,
    )
    n = 0x8AFFFFFFFFFFFEEB0FA77000000089F0DD49FAC5


class Mnt4(WeierstrassInterface):
    a = 0x4BE28760AA064734852CB4FF51EF2928A7A3CD75087C35CB1433714F7407
    b = 0x329704EB1C042F7858C878AA369F70C5C517DE4E05A823DCB8224B8A4D5A
    p = 0xA2FFFFFFFFFFFFFFFFFFFFFFFFC298B00000000000000000000005C866CF
    G = (
        0x82556D57811807A0D7675674B3D57222CFBF9A2A2A2CD146572D7B67627E,
        0x73AFACEA28DC870BAA1D5B0BD4300DDD975E2EEFC7C2DB508FC2E92A8345,
    )
    n = 0xA2FFFFFFFFFFFFFFFFFFFFFFFFC298B00000000000000000000005C866CD


class Mnt5_1(WeierstrassInterface):
    a = 0xD149265D4687DCAB1F2046E0947E51AC5E8E7F25916D35539D4DF2E9017A
    b = 0x489E7783A1F584712BD4F6D48CF2D1CA2C975678936E639083991C5FC369
    p = 0xD2FFFFFFFFFFFFFFFFFFFFFFFE9058D000000000000000000000A0271007
    G = (
        0x1D871A744F1E02ED15D7D84ABD95E80476E6307085F12DBA27092FF06D60,
        0x5C0C8BAE9661303107B0077949DEE16A7F6DDE4982657B9196DE23D9F9D0,
    )
    n = 0xD2FFFFFFFFFFFFFFFFFFFFFFFE9058D000000000000000000000A0271005


class Mnt5_2(WeierstrassInterface):
    a = 0x26CAACED434C5A4C2C9C1B09E0DDC167548A95516E7C81B20702485C9809
    b = 0x6031C89E2CDD91881DBD675BEAC3F3DF8DB1B8E0F45301215A01BAF56AB3
    p = 0xD2FFFFFFFFFFFFFFFFFFFFFFFE9058D000000000000000000000A0271007
    G = (
        0x16E55A2EF696238A7AAF19E51B6A81E1582F28B4BCB6575AB4E0331E569B,
        0x38DE9844643FC9DB3C568EC528983DA16A177D56145A1D4BF88A2340D839,
    )
    n = 0xD2FFFFFFFFFFFFFFFFFFFFFFFE9058D000000000000000000000A0271005


class Mnt5_3(WeierstrassInterface):
    a = 0x44CFC0F3BC92EC82F818B443B564CF25DEE3EBAE7902E370F9E80283D3BD
    b = 0x2DDFD5F7D30C9DACA565CD8278EDDF6E9497F27450AC97A0A69AAC57E27E
    p = 0xD2FFFFFFFFFFFFFFFFFFFFFFFE9058D000000000000000000000A0271007
    G = (
        0xB071579C8CC322DC7FDCE378E5B539B4B7580823ABA3CFDD6637CBFA0BBB,
        0x15D1B75795732B1E2DB1EFA55CDBB19357E0AA0422CC03B442809339CF02,
    )
    n = 0xD2FFFFFFFFFFFFFFFFFFFFFFFE9058D000000000000000000000A0271005


class Prime192v2(WeierstrassInterface):
    a = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFC
    b = 0xCC22D6DFB95C6B25E49C0D6364A4E5980C393AA21668D953
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFF
    G = (
        0xEEA2BAE7E1497842F2DE7769CFE9C989C072AD696F48034A,
        0x6574D11D69B6EC7A672BB82A083DF2F2B0847DE970B2DE15,
    )
    n = 0xFFFFFFFFFFFFFFFFFFFFFFFE5FB1A724DC80418648D8DD31


class Prime192v3(WeierstrassInterface):
    a = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFC
    b = 0x22123DC2395A05CAA7423DAECCC94760A7D462256BD56916
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFF
    G = (
        0x7D29778100C65A1DA1783716588DCE2B8B4AEE8E228F1896,
        0x38A90F22637337334B49DCB66A6DC8F9978ACA7648A943B0,
    )
    n = 0xFFFFFFFFFFFFFFFFFFFFFFFF7A62D031C83F4294F640EC13


class Prime239v1(WeierstrassInterface):
    a = 0x7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFC
    b = 0x6B016C3BDCF18941D0D654921475CA71A9DB2FB27D1D37796185C2942C0A
    p = 0x7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFF
    G = (
        0x0FFA963CDCA8816CCC33B8642BEDF905C3D358573D3F27FBBD3B3CB9AAAF,
        0x7DEBE8E4E90A5DAE6E4054CA530BA04654B36818CE226B39FCCB7B02F1AE,
    )
    n = 0x7FFFFFFFFFFFFFFFFFFFFFFF7FFFFF9E5E9A9F5D9071FBD1522688909D0B


class Prime239v2(WeierstrassInterface):
    a = 0x7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFC
    b = 0x617FAB6832576CBBFED50D99F0249C3FEE58B94BA0038C7AE84C8C832F2C
    p = 0x7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFF
    G = (
        0x38AF09D98727705120C921BB5E9E26296A3CDCF2F35757A0EAFD87B830E7,
        0x5B0125E4DBEA0EC7206DA0FC01D9B081329FB555DE6EF460237DFF8BE4BA,
    )
    n = 0x7FFFFFFFFFFFFFFFFFFFFFFF800000CFA7E8594377D414C03821BC582063


class Prime239v3(WeierstrassInterface):
    a = 0x7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFC
    b = 0x255705FA2A306654B1F4CB03D6A750A30C250102D4988717D9BA15AB6D3E
    p = 0x7FFFFFFFFFFFFFFFFFFFFFFF7FFFFFFFFFFF8000000000007FFFFFFFFFFF
    G = (
        0x6768AE8E18BB92CFCF005C949AA2C6D94853D0E660BBF854B1C9505FE95A,
        0x1607E6898F390C06BC1D552BAD226F3B6FCFE48B6E818499AF18E3ED6CF3,
    )
    n = 0x7FFFFFFFFFFFFFFFFFFFFFFF7FFFFF975DEB41B3A6057C3C432146526551


class Bls12_377(WeierstrassInterface):
    a = 0x00
    b = 0x01
    p = 0x01AE3A4617C510EAC63B05C06CA1493B1A22D9F300F5138F1EF3622FBA094800170B5D44300000008508C00000000001
    G = (
        0x008848DEFE740A67C8FC6225BF87FF5485951E2CAA9D41BB188282C8BD37CB5CD5481512FFCD394EEAB9B16EB21BE9EF,
        0x01914A69C5102EFF1F674F5D30AFEEC4BD7FB348CA3E52D96D182AD44FB82305C2FE3D3634A9591AFD82DE55559C8EA6,
    )
    n = 0x12AB655E9A2CA55660B44D1E5C37B00159AA76FED00000010A11800000000001


class Bls12_381(WeierstrassInterface):
    a = 0x00
    b = 0x04
    p = 0x1A0111EA397FE69A4B1BA7B6434BACD764774B84F38512BF6730D2A0F6B0F6241EABFFFEB153FFFFB9FEFFFFFFFFAAAB
    G = (
        0x17F1D3A73197D7942695638C4FA9AC0FC3688C4F9774B905A14E3A3F171BAC586C55E83FF97A1AEFFB3AF00ADB22C6BB,
        0x08B3F481E3AAA0F1A09E30ED741D8AE4FCF5E095D5D00AF600DB18CB2C04B3EDD03CC744A2888AE40CAA232946C5E7E1,
    )
    n = 0x73EDA753299D7D483339D80809A1D80553BDA402FFFE5BFEFFFFFFFF00000001


class Bls12_446(WeierstrassInterface):
    a = 0x00
    b = 0x01
    p = 0x3CDEE0FB28C5E535200FC34965AAD6400095A4B78A02FE320F75A64BBAC71602824E6DC3E23ACDEE56EE4528C573B5CC311C0026AAB0AAAB
    G = (
        0x297792B2D03DE39D64FACA6D49FCF7A8850144CA24FC5D815C082A3AA87D1A16929E56228C136123BA51421AE89CACD5B4789A38CE39035A,
        0xDC40DDCBAB2823A7870B5C688AA04FEE40369D913E4F2F0947A152FE1C27A79B7F787E9C35B869C3846FAC4F12A70D0FE22D2E244268CC,
    )
    n = 0x511B70539F27995B34995830FA4D04C98CCC4C050BC7BB9B0E8D8CA34610428001400040001


class Bls12_455(WeierstrassInterface):
    a = 0x00
    b = 0x0A
    p = 0x55555955557955572AA00E0F95B49203003F665E3A5B1D56234BD93954FCB314B8B3DB9994ACE86D1BA6C589556B2AA956AAA00001800002AB
    G = (
        0x03018DF4C2336D178E6EA61540353ABA01923E3890B3295848906DFC90E0C43008E4751FFC913DC1FF3DF33D11DB57BADA7A9297195ACFB2FA,
        0x19A8A9C4C3AC2FFB4C6B380D17B8282E029615052EAA6416C16C8F36F251D87C272657F0702CC58C4E072628D7BAD3C0E9B3A8AEBFC6B2357C,
    )
    n = 0x10000080000380002E0000F10004F00025E000750001D1000A00000400001C00007FFFFC00001


class Bls12_638(WeierstrassInterface):
    a = 0x00
    b = 0x04
    p = 0x3CB868653D300B3FE80015554DD25DB0FC01DCDE95D4000000631BBD421715013955555555529C005C75D6C2AB00000000000AC79600D2ABAAAAAAAAAAAAAA93EAF3FF000AAAAAAAAAAAAAAABEAB000B
    G = (
        0x160F63A3A3B297F113075ED79466138E85B025F7FE724B78E32D7AFC4D734BDD54F871092B8D1966D491C0F45A48A8BBA5586095DFFCC1410B7E26ED16BAF98C1117959134C24A17A7BE31E1AFBF844F,
        0x2D340B33877480A9785E86ED2EDCAFC170B82568CB21B708B79FC6DA3748461FCD80697E486695F3CAE76FCB1781E784F6812F57BE05DFC850426650DED8B40A464B00A35718228EC8E02B52B59D876E,
    )
    n = 0x50F94035FF4000FFFFFFFFFFF9406BFDC0040000000000000035FB801DFFBFFFFFFFFFFFFFFF401BFF80000000000000000000FFC01


class Bls24_477(WeierstrassInterface):
    a = 0x00
    b = 0x04
    p = 0x167278FAC63BD5B007EBB8F693A2AB3DBD9F92CF437C399D928E94BFE9A04A009FDA9E8CF9226901DE62AEA9DCEA48BF1A0EBBF8860A5E7AD000152B
    G = (
        0x15DFD8E4893A408A34B600532B51CC86CAB3AF07103CFCF3EC7B9AF836904CFB60AB0FA8AC91EE6255E5EF6286FA0C24DF9D76EA50599C2E103E40AD,
        0x0A683957A59B1B488FA657E11B44815056BDE33C09D6AAD392D299F89C7841B91A683BF01B7E70547E48E0FBE1CA9E991983131470F886BA9B6FCE2E,
    )
    n = 0x57F52EE445CC41781FCD53D13E45F6ACDFE4F9F2A3CD414E71238AFC9FCFC7D38CAEF64F4FF79F90013FFFFFF0000001


class Gost256(WeierstrassInterface):
    a = 0x7
    b = 0x5FBFF498AA938CE739B8E022FBAFEF40563F6E6A3472FC2A514C0CE9DAE23B7E
    p = 0x8000000000000000000000000000000000000000000000000000000000000431
    G = (0x2, 0x8E2A8A0E65147D4BD6316030E16D19C85C97F0A9CA267122B96ABBCEA7E8FC8)
    n = 0x8000000000000000000000000000000150FE8A1892976154C59CFC193ACCF5B3


class Gost512(WeierstrassInterface):
    a = 0x7
    b = 0x1CFF0806A31116DA29D8CFA54E57EB748BC5F377E49400FDD788B649ECA1AC4361834013B2AD7322480A89CA58E0CF74BC9E540C2ADD6897FAD0A3084F302ADC
    p = 0x4531ACD1FE0023C7550D267B6B2FEE80922B14B2FFB90F04D4EB7C09B5D2D15DF1D852741AF4704A0458047E80E4546D35B8336FAC224DD81664BBF528BE6373
    G = (
        0x24D19CC64572EE30F396BF6EBBFD7A6C5213B3B3D7057CC825F91093A68CD762FD60611262CD838DC6B60AA7EEE804E28BC849977FAC33B4B530F1B120248A9A,
        0x2BB312A43BD2CE6E0D020613C857ACDDCFBF061E91E5F2C3F32447C259F39B2C83AB156D77F1496BF7EB3351E1EE4E43DC1A18B91B24640B6DBB92CB1ADD371E,
    )
    n = 0x4531ACD1FE0023C7550D267B6B2FEE80922B14B2FFB90F04D4EB7C09B5D2D15DA82F2D7ECB1DBAC719905C5EECC423F1D86E25EDBE23C595D644AAF187E6E6DF


class Bn158(WeierstrassInterface):
    a = 0x0000000000000000000000000000000000000000
    b = 0x0000000000000000000000000000000000000011
    p = 0x24240D8241D5445106C8442084001384E0000013
    G = (
        0x24240D8241D5445106C8442084001384E0000012,
        0x0000000000000000000000000000000000000004,
    )
    n = 0x24240D8241D5445106C7E3F07E0010842000000D


class Bn190(WeierstrassInterface):
    a = 0x000000000000000000000000000000000000000000000000
    b = 0x000000000000000000000000000000000000000000001001
    p = 0x240001B0000948001E60004134005F10005DC0003A800013
    G = (
        0x240001B0000948001E60004134005F10005DC0003A800012,
        0x000000000000000000000000000000000000000000000040,
    )
    n = 0x240001B0000948001E600040D4005CD0005760003180000D


class Bn222(WeierstrassInterface):
    a = 0x00000000000000000000000000000000000000000000000000000000
    b = 0x00000000000000000000000000000000000000000000000000000101
    p = 0x23DC0D7DC02402CDE486F4C00015B5215C0000004C6CE00000000067
    G = (
        0x23DC0D7DC02402CDE486F4C00015B5215C0000004C6CE00000000066,
        0x00000000000000000000000000000000000000000000000000000010,
    )
    n = 0x23DC0D7DC02402CDE486F4C00015555156000000496DA00000000061


class Bn254(WeierstrassInterface):
    a = 0x0000000000000000000000000000000000000000000000000000000000000000
    b = 0x0000000000000000000000000000000000000000000000000000000000000002
    p = 0x2523648240000001BA344D80000000086121000000000013A700000000000013
    G = (
        0x2523648240000001BA344D80000000086121000000000013A700000000000012,
        0x0000000000000000000000000000000000000000000000000000000000000001,
    )
    n = 0x2523648240000001BA344D8000000007FF9F800000000010A10000000000000D


class Bn286(WeierstrassInterface):
    a = 0x000000000000000000000000000000000000000000000000000000000000000000000000
    b = 0x000000000000000000000000000000000000000000000000000000000000000000000002
    p = 0x240900D8991B25B0E2CB51DDA534A205391892080A008108000853813800138000000013
    G = (
        0x240900D8991B25B0E2CB51DDA534A205391892080A008108000853813800138000000012,
        0x000000000000000000000000000000000000000000000000000000000000000000000001,
    )
    n = 0x240900D8991B25B0E2CB51DDA534A205391831FC099FC0FC0007F081080010800000000D


class Bn318(WeierstrassInterface):
    a = 0x00000000000000000000000000000000000000000000000000000000000000000000000000000000
    b = 0x00000000000000000000000000000000000000000000000000000000000000000000000000000002
    p = 0x24009000D800900024075015F015F0075000008F411E808F4000000004E484E4800000000000101B
    G = (
        0x24009000D800900024075015F015F0075000008F411E808F4000000004E484E4800000000000101A,
        0x00000000000000000000000000000000000000000000000000000000000000000000000000000001,
    )
    n = 0x24009000D800900024075015F015F0075000008EE11DC08EE000000004DB84DB8000000000000FE5


class Bn350(WeierstrassInterface):
    a = 0x0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    b = 0x0000000000000000000000000000000000000000000000000000000000000000000000000000000000000002
    p = 0x23FFB80035FFEE24020A01CAFD738EC3F24B475EBC0AD0F6A0530FD78443FDF01A3FF64084000004E0000013
    G = (
        0x23FFB80035FFEE24020A01CAFD738EC3F24B475EBC0AD0F6A0530FD78443FDF01A3FF64084000004E0000012,
        0x0000000000000000000000000000000000000000000000000000000000000000000000000000000000000001,
    )
    n = 0x23FFB80035FFEE24020A01CAFD738EC3F24B475EBC0A70F70052F7D78413FE08173FF7C07E0000042000000D


class Bn382(WeierstrassInterface):
    a = 0x000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    b = 0x000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002
    p = 0x240026400F3D82B2E42DE125B00158405B710818AC00000840046200950400000000001380052E000000000000000013
    G = (
        0x240026400F3D82B2E42DE125B00158405B710818AC00000840046200950400000000001380052E000000000000000012,
        0x000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001,
    )
    n = 0x240026400F3D82B2E42DE125B00158405B710818AC000007E0042F008E3E00000000001080046200000000000000000D


class Bn414(WeierstrassInterface):
    a = 0x00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    b = 0x00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002
    p = 0x240024000D7EE23F2823CA035D31B144364C75E59AEFFF60544845142000765EFFF7C0000021138004DFFFFFD900000000000013
    G = (
        0x240024000D7EE23F2823CA035D31B144364C75E59AEFFF60544845142000765EFFF7C0000021138004DFFFFFD900000000000012,
        0x00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001,
    )
    n = 0x240024000D7EE23F2823CA035D31B144364C75E59AEFFF605447E513F00070607FF82000001F9080041FFFFFDF0000000000000D


class Bn446(WeierstrassInterface):
    a = 0x0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    b = 0x0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000101
    p = 0x2400000000000000002400000002D00000000D800000021C0000001800000000870000000B0400000057C00000015C000000132000000067
    G = (
        0x2400000000000000002400000002D00000000D800000021C0000001800000000870000000B0400000057C00000015C000000132000000066,
        0x0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010,
    )
    n = 0x2400000000000000002400000002D00000000D800000021C00000017A0000000870000000AD400000054C000000156000000126000000061


class Bn478(WeierstrassInterface):
    a = 0x000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    b = 0x000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002
    p = 0x23FFFFFFFFFFFFFEDFFFFFFFEE0001B3600000006BFFF5DB835FFF5D28085442328002888F96F2944D7DED781430FFD780065FFF010020FFFD900013
    G = (
        0x23FFFFFFFFFFFFFEDFFFFFFFEE0001B3600000006BFFF5DB835FFF5D28085442328002888F96F2944D7DED781430FFD780065FFF010020FFFD900012,
        0x000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001,
    )
    n = 0x23FFFFFFFFFFFFFEDFFFFFFFEE0001B3600000006BFFF5DB835FFF5D2807F442328002888F9872944D7E0578112F7FD780062FFF07001F7FFDF0000D


class Bn510(WeierstrassInterface):
    a = 0x00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    b = 0x00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000101
    p = 0x2400000000000000003F000000000001B0002958000000000237000C0F0000084000F8100151A400073800242D00001380019440000000000888000000000013
    G = (
        0x2400000000000000003F000000000001B0002958000000000237000C0F0000084000F8100151A400073800242D00001380019440000000000888000000000012,
        0x00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010,
    )
    n = 0x2400000000000000003F000000000001B0002958000000000237000C0F000007E000F8100151A40006E400242D000010800181E000000000073800000000000D


class Bn542(WeierstrassInterface):
    a = 0x0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    b = 0x0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002
    p = 0x2400090000D80009000024000090001B01B1B051090510001B00D8001B0510D8A2084511080008D000090510005110800108138025380001B00000084000001380000013
    G = (
        0x2400090000D80009000024000090001B01B1B051090510001B00D8001B0510D8A2084511080008D000090510005110800108138025380001B00000084000001380000012,
        0x0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001,
    )
    n = 0x2400090000D80009000024000090001B01B1B051090510001B00D8001B0510D8A207E510FC0008700009051000510FC000FC108025080001B0000007E00000108000000D


class Bn574(WeierstrassInterface):
    a = 0x000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    b = 0x000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002
    p = 0x2400023FFFFB7FFF4C00002400167FFFEE01AEE014423FAEFFFB5C000A200050FFFF2808400041FFFE73FFF7C000210000000000001380004DFFFD90000000000000000000000013
    G = (
        0x2400023FFFFB7FFF4C00002400167FFFEE01AEE014423FAEFFFB5C000A200050FFFF2808400041FFFE73FFF7C000210000000000001380004DFFFD90000000000000000000000012,
        0x000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001,
    )
    n = 0x2400023FFFFB7FFF4C00002400167FFFEE01AEE014423FAEFFFB5C000A200050FFFF2807E0003EFFFE85FFF820001F80000000000010800041FFFDF000000000000000000000000D


class Bn606(WeierstrassInterface):
    a = 0x00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    b = 0x00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002
    p = 0x23FFFFFFFFFFFEE00000000000036000000241AFFB7FFFFFF275E0024000001B1440000D94482DF27FFFC9AEDF0000000036512100245142137FFFFFB75D7BD900000000000000246C844E13
    G = (
        0x23FFFFFFFFFFFEE00000000000036000000241AFFB7FFFFFF275E0024000001B1440000D94482DF27FFFC9AEDF0000000036512100245142137FFFFFB75D7BD900000000000000246C844E12,
        0x00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001,
    )
    n = 0x23FFFFFFFFFFFEE00000000000036000000241AFFB7FFFFFF275E0024000001B1440000D9447CDF27FFFC9AEE08000000036511F8024513F107FFFFFB75D81DF00000000000000246C7E420D


class Bn638(WeierstrassInterface):
    a = 0x0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
    b = 0x0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000101
    p = 0x23FFFFFDC000000D7FFFFFB8000001D3FFFFF942D000165E3FFF94870000D52FFFFDD0E00008DE55C00086520021E55BFFFFF51FFFF4EB800000004C80015ACDFFFFFFFFFFFFECE00000000000000067
    G = (
        0x23FFFFFDC000000D7FFFFFB8000001D3FFFFF942D000165E3FFF94870000D52FFFFDD0E00008DE55C00086520021E55BFFFFF51FFFF4EB800000004C80015ACDFFFFFFFFFFFFECE00000000000000066,
        0x0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010,
    )
    n = 0x23FFFFFDC000000D7FFFFFB8000001D3FFFFF942D000165E3FFF94870000D52FFFFDD0E00008DE55600086550021E555FFFFF54FFFF4EAC000000049800154D9FFFFFFFFFFFFEDA00000000000000061


class Secp112r1(WeierstrassInterface):
    a = 0xDB7C2ABF62E35E668076BEAD2088
    b = 0x659EF8BA043916EEDE8911702B22
    p = 0xDB7C2ABF62E35E668076BEAD208B
    G = (0x09487239995A5EE76B55F9C2F098, 0xA89CE5AF8724C0A23E0E0FF77500)
    n = 0xDB7C2ABF62E35E7628DFAC6561C5


class Secp112r2(WeierstrassInterface):
    a = 0x6127C24C05F38A0AAAF65C0EF02C
    b = 0x51DEF1815DB5ED74FCC34C85D709
    p = 0xDB7C2ABF62E35E668076BEAD208B
    G = (0x4BA30AB5E892B4E1649DD0928643, 0xADCD46F5882E3747DEF36E956E97)
    n = 0x36DF0AAFD8B8D7597CA10520D04B


class Secp128r1(WeierstrassInterface):
    a = 0xFFFFFFFDFFFFFFFFFFFFFFFFFFFFFFFC
    b = 0xE87579C11079F43DD824993C2CEE5ED3
    p = 0xFFFFFFFDFFFFFFFFFFFFFFFFFFFFFFFF
    G = (0x161FF7528B899B2D0C28607CA52C5B86, 0xCF5AC8395BAFEB13C02DA292DDED7A83)
    n = 0xFFFFFFFE0000000075A30D1B9038A115


class Secp128r2(WeierstrassInterface):
    a = 0xD6031998D1B3BBFEBF59CC9BBFF9AEE1
    b = 0x5EEEFCA380D02919DC2C6558BB6D8A5D
    p = 0xFFFFFFFDFFFFFFFFFFFFFFFFFFFFFFFF
    G = (0x7B6AA5D85E572983E6FB32A7CDEBC140, 0x27B6916A894D3AEE7106FE805FC34B44)
    n = 0x3FFFFFFF7FFFFFFFBE0024720613B5A3


class Secp160k1(WeierstrassInterface):
    a = 0x0000000000000000000000000000000000000000
    b = 0x0000000000000000000000000000000000000007
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFAC73
    G = (
        0x3B4C382CE37AA192A4019E763036F4F5DD4D7EBB,
        0x938CF935318FDCED6BC28286531733C3F03C4FEE,
    )
    n = 0x0100000000000000000001B8FA16DFAB9ACA16B6B3


class Secp160r1(WeierstrassInterface):
    a = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7FFFFFFC
    b = 0x1C97BEFC54BD7A8B65ACF89F81D4D4ADC565FA45
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7FFFFFFF
    G = (
        0x4A96B5688EF573284664698968C38BB913CBFC82,
        0x23A628553168947D59DCC912042351377AC5FB32,
    )
    n = 0x0100000000000000000001F4C8F927AED3CA752257


class Secp160r2(WeierstrassInterface):
    a = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFAC70
    b = 0xB4E134D3FB59EB8BAB57274904664D5AF50388BA
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFAC73
    G = (
        0x52DCB034293A117E1F4FF11B30F7199D3144CE6D,
        0xFEAFFEF2E331F296E071FA0DF9982CFEA7D43F2E,
    )
    n = 0x0100000000000000000000351EE786A818F3A1A16B


class Secp192k1(WeierstrassInterface):
    a = 0x000000000000000000000000000000000000000000000000
    b = 0x000000000000000000000000000000000000000000000003
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFEE37
    G = (
        0xDB4FF10EC057E9AE26B07D0280B7F4341DA5D1B1EAE06C7D,
        0x9B2F2F6D9C5628A7844163D015BE86344082AA88D95E2F9D,
    )
    n = 0xFFFFFFFFFFFFFFFFFFFFFFFE26F2FC170F69466A74DEFD8D


class Secp224k1(WeierstrassInterface):
    a = 0x00000000000000000000000000000000000000000000000000000000
    b = 0x00000000000000000000000000000000000000000000000000000005
    p = 0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFE56D
    G = (
        0xA1455B334DF099DF30FC28A169A467E9E47075A90F7E650EB6B7A45C,
        0x7E089FED7FBA344282CAFBD6F7E319F7C0B0BD59E2CA4BDB556D61A5,
    )
    n = 0x10000000000000000000000000001DCE8D2EC6184CAF0A971769FB1F7


class EccFrog512CK2(WeierstrassInterface):
    """
    github.com/victormeloasm/OpenFrogget
    """

    a = -7
    b = 95864189850957917703933006131793785649240252916618759767550461391845895018181
    p = 9149012705592502490164965176888130701548053918699793689672344807772801105830681498780746622530729418858477103073591918058480028776841126664954537807339721
    G = (
        8426241697659200371183582771153260966569955699615044232640972423431947060129573736112298744977332416175021337082775856058058394786264506901662703740544432,
        4970129934163735248083452609809843496231929620419038489506391366136186485994288320758668172790060801809810688192082146431970683113557239433570011112556001,
    )
    n = 9149012705592502490164965176888130701548053918699793689672344807772801105830557269123255850915745063541133157503707284048429261692283957712127567713136519


class SM2(WeierstrassInterface):
    a = 0xFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000FFFFFFFFFFFFFFFC
    b = 0x28E9FA9E9D9F5E344D5A9E4BCF6509A7F39789F515AB8F92DDBCBD414D940E93
    p = 0xFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000FFFFFFFFFFFFFFFF
    G = (
        0x32C4AE2C1F1981195F9904466A39C9948FE30BBFF2660BE1715A4589334C74C7,
        0xBC3736A2F4F6779C59BDCEE36B692153D0A9877CC62A474002DF32E52139F0A0,
    )
    n = 0xFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFF7203DF6B21C6052B53BBF40939D54123


class Test_Curve(WeierstrassInterface):
    """
    koclab.cs.ucsb.edu/teaching/ccs130h/2018/04ecc.pdf
    """

    a = 1
    b = 1
    p = 23
    G = (0, 1)
    n = 28

    def __init__(self):
        logger.warn(
            "weierstrass test-curve is for development and educational purposes only"
            " and should not be used in production."
        )
